<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Models\Vessel;
use App\Models\Certificate;
use App\Models\VesselCertificate;

class ImportCertificatesFromSql extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'certificates:import-from-sql';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import certificates from the aikhrrte_certificates.sql file';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting certificate import from SQL file...');

        try {
            // Get all vessels and certificates
            $vessels = Vessel::all();
            $certificates = Certificate::all();

            // Get existing vessel certificates to avoid duplicates
            $existingCertificates = VesselCertificate::all();
            $existingCount = $existingCertificates->count();
            $this->info("Found {$existingCount} existing vessel certificates");

            // Get vessel certificates from SQL file
            $sqlFilePath = base_path('aikhrrte_certificates.sql');
            
            if (!file_exists($sqlFilePath)) {
                $this->error("SQL file not found at: {$sqlFilePath}");
                return 1;
            }

            $sqlContent = file_get_contents($sqlFilePath);
            
            // Extract vessel_certificates data
            preg_match('/INSERT INTO `vessel_certificates` \(`id`, `vessel_id`, `certificate_id`, `id_no`, `vessel_department`, `issued_date`, `expiry_date`, `attachment_path`, `created_at`, `updated_at`\) VALUES(.*?);/s', $sqlContent, $matches);
            
            if (empty($matches)) {
                $this->error("No vessel_certificates data found in SQL file");
                return 1;
            }

            $valuesString = $matches[1];
            
            // Parse values
            preg_match_all('/\((.*?)\)(?:,|$)/s', $valuesString, $valueMatches);
            $values = $valueMatches[1];
            
            $importCount = 0;
            $skipCount = 0;
            
            // Begin transaction
            DB::beginTransaction();
            
            foreach ($values as $value) {
                $fields = str_getcsv($value, ',', "'");
                
                // Clean up fields
                foreach ($fields as &$field) {
                    $field = trim($field);
                    if ($field === 'NULL' || $field === 'null') {
                        $field = null;
                    }
                }
                
                // Extract data
                $id = (int)$fields[0];
                $vesselId = (int)$fields[1];
                $certificateId = (int)$fields[2];
                $idNo = $fields[3];
                $vesselDepartment = $fields[4];
                $issuedDate = $fields[5] !== null ? $fields[5] : null;
                $expiryDate = $fields[6] !== null ? $fields[6] : null;
                $attachmentPath = $fields[7];
                
                // Check if vessel and certificate exist
                if (!$vessels->contains('id', $vesselId) || !$certificates->contains('id', $certificateId)) {
                    $this->warn("Skipping record with vessel_id: {$vesselId}, certificate_id: {$certificateId} - not found in database");
                    $skipCount++;
                    continue;
                }
                
                // Check if certificate already exists for this vessel
                if ($existingCertificates->where('vessel_id', $vesselId)->where('certificate_id', $certificateId)->count() > 0) {
                    $this->line("Skipping duplicate certificate for vessel_id: {$vesselId}, certificate_id: {$certificateId}");
                    $skipCount++;
                    continue;
                }
                
                // Create new vessel certificate
                VesselCertificate::create([
                    'vessel_id' => $vesselId,
                    'certificate_id' => $certificateId,
                    'id_no' => $idNo,
                    'vessel_department' => $vesselDepartment,
                    'issued_date' => $issuedDate,
                    'expiry_date' => $expiryDate,
                    'attachment_path' => $attachmentPath,
                ]);
                
                $importCount++;
            }
            
            // Commit transaction
            DB::commit();
            
            $this->info("Import completed successfully!");
            $this->info("Imported {$importCount} certificates");
            $this->info("Skipped {$skipCount} certificates");
            
            return 0;
        } catch (\Exception $e) {
            // Rollback transaction on error
            DB::rollBack();
            
            $this->error("Error during import: " . $e->getMessage());
            Log::error("Certificate import error: " . $e->getMessage());
            Log::error($e->getTraceAsString());
            
            return 1;
        }
    }
}
