<?php

namespace App\Http\Controllers;

use App\Models\Certificate;
use App\Models\Survey;
use App\Models\Vessel;
use App\Models\VesselCertificate;
use App\Models\WarningSettings;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class SettingsController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display the settings page.
     */
    public function index()
    {
        $settings = WarningSettings::first() ?? new WarningSettings();

        // Get all certificates for the certificates tab
        $certificates = Certificate::orderBy('category')
                                 ->orderBy('display_order')
                                 ->get();

        // Get all categories for filter dropdown
        $categories = Certificate::select('category')
                                ->distinct()
                                ->orderBy('category')
                                ->pluck('category');

        return view('settings.index', compact('settings', 'certificates', 'categories'));
    }

    /**
     * Update the warning settings.
     */
    public function updateWarningSettings(Request $request)
    {
        $request->validate([
            'warning_days' => 'required|integer|min:1',
            'danger_days' => 'required|integer|min:1',
            'survey_warning_days' => 'required|integer|min:1',
            'survey_danger_days' => 'required|integer|min:1',
        ]);

        $settings = WarningSettings::first();
        if (!$settings) {
            $settings = new WarningSettings();
        }

        $settings->warning_days = $request->warning_days;
        $settings->danger_days = $request->danger_days;
        $settings->survey_warning_days = $request->survey_warning_days;
        $settings->survey_danger_days = $request->survey_danger_days;
        $settings->save();

        return redirect()->route('settings.index')
            ->with('success', 'Warning settings updated successfully.');
    }

    /**
     * Show the certificates management page.
     */
    public function certificates()
    {
        $certificates = Certificate::orderBy('category')
                                 ->orderBy('display_order')
                                 ->get();

        // Get all categories for filter dropdown
        $categories = Certificate::select('category')
                                ->distinct()
                                ->orderBy('category')
                                ->pluck('category');

        return view('settings.certificates', compact('certificates', 'categories'));
    }

    /**
     * Create a new certificate and add it to all vessels.
     */
    public function createCertificate(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:100|unique:certificates',
            'category' => 'required|string|max:100',
        ]);

        // Begin transaction
        DB::beginTransaction();

        try {
            // Create new certificate
            $certificate = Certificate::create([
                'name' => $request->name,
                'category' => $request->category,
                'display_order' => Certificate::where('category', $request->category)->max('display_order') + 1,
            ]);

            // Always add to all vessels
            $vessels = Vessel::all();
            $addedCount = 0;
            foreach ($vessels as $vessel) {
                VesselCertificate::create([
                    'vessel_id' => $vessel->id,
                    'certificate_id' => $certificate->id,
                ]);
                $addedCount++;
            }

            // Commit transaction
            DB::commit();

            return redirect()->route('settings.certificates')
                ->with('success', "Certificate '{$certificate->name}' created successfully and added to {$addedCount} vessels.");

        } catch (\Exception $e) {
            // Rollback transaction on error
            DB::rollBack();

            return redirect()->route('settings.certificates')
                ->with('error', 'Error creating certificate: ' . $e->getMessage());
        }
    }

    /**
     * Update an existing certificate.
     */
    public function updateCertificate(Request $request, Certificate $certificate)
    {
        $request->validate([
            'name' => 'required|string|max:100|unique:certificates,name,' . $certificate->id,
            'category' => 'required|string|max:100',
        ]);

        // Begin transaction
        DB::beginTransaction();

        try {
            // Update certificate
            $certificate->name = $request->name;
            $certificate->category = $request->category;
            $certificate->save();

            // Commit transaction
            DB::commit();

            return redirect()->route('settings.certificates')
                ->with('success', "Certificate '{$certificate->name}' updated successfully.");

        } catch (\Exception $e) {
            // Rollback transaction on error
            DB::rollBack();

            return redirect()->route('settings.certificates')
                ->with('error', 'Error updating certificate: ' . $e->getMessage());
        }
    }

    /**
     * Delete a certificate and remove it from all vessels.
     */
    public function destroyCertificate(Request $request, Certificate $certificate)
    {
        // Begin transaction
        DB::beginTransaction();

        try {
            // Get the certificate name for the success message
            $certificateName = $certificate->name;

            // Delete all vessel certificates associated with this certificate
            VesselCertificate::where('certificate_id', $certificate->id)->delete();

            // Delete the certificate
            $certificate->delete();

            // Commit transaction
            DB::commit();

            // Check if it's an AJAX request
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => "Certificate '{$certificateName}' deleted successfully and removed from all vessels."
                ]);
            }

            return redirect()->route('settings.certificates')
                ->with('success', "Certificate '{$certificateName}' deleted successfully and removed from all vessels.");

        } catch (\Exception $e) {
            // Rollback transaction on error
            DB::rollBack();

            // Check if it's an AJAX request
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error deleting certificate: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->route('settings.certificates')
                ->with('error', 'Error deleting certificate: ' . $e->getMessage());
        }
    }

    /**
     * Reorder certificates.
     */
    public function reorderCertificates(Request $request)
    {
        $request->validate([
            'certificates' => 'required|array',
            'certificates.*.id' => 'required|exists:certificates,id',
            'certificates.*.display_order' => 'required|integer|min:0',
        ]);

        // Begin transaction
        DB::beginTransaction();

        try {
            foreach ($request->certificates as $cert) {
                $certificate = Certificate::findOrFail($cert['id']);
                $certificate->display_order = $cert['display_order'];
                $certificate->save();
            }

            // Commit transaction
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Certificates reordered successfully.']);

        } catch (\Exception $e) {
            // Rollback transaction on error
            DB::rollBack();

            return response()->json(['success' => false, 'message' => 'Error reordering certificates: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Show the database management page.
     */
    public function database(Request $request)
    {
        // Check if user is admin
        if (!auth()->user()->is_admin) {
            abort(403, 'Unauthorized action. Admin access required.');
        }

        $query = VesselCertificate::with(['vessel', 'certificate']);

        // Apply filters
        if ($request->filled('vessel_id')) {
            $query->where('vessel_id', $request->vessel_id);
        }

        if ($request->filled('certificate_id')) {
            $query->where('certificate_id', $request->certificate_id);
        }

        if ($request->filled('status')) {
            switch ($request->status) {
                case 'expired':
                    $query->whereNotNull('expiry_date')->whereDate('expiry_date', '<', now());
                    break;
                case 'expiring_soon':
                    $query->whereNotNull('expiry_date')
                          ->whereDate('expiry_date', '>=', now())
                          ->whereDate('expiry_date', '<=', now()->addDays(30));
                    break;
                case 'valid':
                    $query->whereNotNull('expiry_date')->whereDate('expiry_date', '>', now()->addDays(30));
                    break;
                case 'no_expiry':
                    $query->whereNull('expiry_date');
                    break;
            }
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('id_no', 'like', "%{$search}%")
                  ->orWhereHas('vessel', function($vq) use ($search) {
                      $vq->where('name', 'like', "%{$search}%");
                  })
                  ->orWhereHas('certificate', function($cq) use ($search) {
                      $cq->where('name', 'like', "%{$search}%");
                  });
            });
        }

        // Check for duplicates
        if ($request->filled('show_duplicates')) {
            $duplicateIds = VesselCertificate::select('vessel_id', 'certificate_id')
                ->selectRaw('COUNT(*) as count')
                ->groupBy('vessel_id', 'certificate_id')
                ->having('count', '>', 1)
                ->get()
                ->map(function($item) {
                    return VesselCertificate::where('vessel_id', $item->vessel_id)
                        ->where('certificate_id', $item->certificate_id)
                        ->pluck('id');
                })
                ->flatten();

            $query->whereIn('id', $duplicateIds);
        }

        $vesselCertificates = $query->orderBy('vessel_id')
                                   ->orderBy('certificate_id')
                                   ->orderBy('expiry_date', 'desc')
                                   ->get();

        // Get filter options
        $vessels = Vessel::orderBy('name')->get();
        $certificates = Certificate::orderBy('category')->orderBy('name')->get();

        // Get duplicate statistics
        $duplicateStats = VesselCertificate::select('vessel_id', 'certificate_id')
            ->selectRaw('COUNT(*) as count')
            ->groupBy('vessel_id', 'certificate_id')
            ->having('count', '>', 1)
            ->with(['vessel', 'certificate'])
            ->get();

        return view('settings.database', compact(
            'vesselCertificates',
            'vessels',
            'certificates',
            'duplicateStats'
        ));
    }

    /**
     * Show the surveys management page.
     */
    public function surveys()
    {
        // Check if user is admin
        if (!auth()->user()->is_admin) {
            abort(403, 'Unauthorized action. Admin access required.');
        }

        $surveys = Survey::orderBy('category')->orderBy('name')->get();

        // Get all categories for filter dropdown
        $categories = Survey::select('category')
                           ->distinct()
                           ->orderBy('category')
                           ->pluck('category');

        return view('settings.surveys', compact('surveys', 'categories'));
    }

    /**
     * Create a new survey.
     */
    public function createSurvey(Request $request)
    {
        // Check if user is admin
        if (!auth()->user()->is_admin) {
            abort(403, 'Unauthorized action. Admin access required.');
        }

        $request->validate([
            'name' => 'required|string|max:255|unique:surveys',
            'category' => 'required|string|max:255',
        ]);

        Survey::create([
            'name' => $request->name,
            'category' => $request->category,
        ]);

        return redirect()->route('settings.surveys')
            ->with('success', 'Survey created successfully.');
    }

    /**
     * Update an existing survey.
     */
    public function updateSurvey(Request $request, $id)
    {
        // Check if user is admin
        if (!auth()->user()->is_admin) {
            abort(403, 'Unauthorized action. Admin access required.');
        }

        $survey = Survey::findOrFail($id);

        $request->validate([
            'name' => 'required|string|max:255|unique:surveys,name,' . $survey->id,
            'category' => 'required|string|max:255',
        ]);

        $survey->update([
            'name' => $request->name,
            'category' => $request->category,
        ]);

        return redirect()->route('settings.surveys')
            ->with('success', 'Survey updated successfully.');
    }

    /**
     * Delete a survey.
     */
    public function destroySurvey($id)
    {
        // Check if user is admin
        if (!auth()->user()->is_admin) {
            abort(403, 'Unauthorized action. Admin access required.');
        }

        $survey = Survey::findOrFail($id);

        // Check if survey is being used by any vessels
        if ($survey->vesselSurveys()->count() > 0) {
            return redirect()->route('settings.surveys')
                ->with('error', 'Cannot delete survey. It is currently assigned to ' . $survey->vesselSurveys()->count() . ' vessel(s).');
        }

        $survey->delete();

        return redirect()->route('settings.surveys')
            ->with('success', 'Survey deleted successfully.');
    }

}
