/*
 * Copyright 2015 Google Inc. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

if (!('inert' in HTMLElement.prototype)) {
  Object.defineProperty(HTMLElement.prototype, 'inert', {
    enumerable: true,
    get: function() { return this.hasAttribute('inert'); },
    set: function(inert) {
      if (inert) {
        this.setAttribute('inert', '');
      } else {
        this.removeAttribute('inert');
      }
    }
  });

  // Autofocus elements can't be inert
  function observeAutofocus(element) {
    if (element.autofocus) {
      console.warn('Element', element, 'has autofocus and inert, which is a contradiction. Removing autofocus.');
      element.removeAttribute('autofocus');
    }
  }

  function recursivelyCheckForInert(element) {
    if (element.hasAttribute('inert')) {
      const inertRoot = element;
      
      // Everything inside an inert element is inert
      const elementsToMakeInert = Array.from(element.querySelectorAll('*'));
      
      // Make all elements inside inert
      elementsToMakeInert.forEach(el => {
        // Skip elements that are already inert
        if (el.hasAttribute('inert')) {
          return;
        }
        
        // Save the original tabindex and disabled state
        if (el.hasAttribute('tabindex')) {
          el.setAttribute('data-original-tabindex', el.getAttribute('tabindex'));
        }
        if (el.hasAttribute('disabled')) {
          el.setAttribute('data-original-disabled', 'true');
        }
        
        // Make the element unfocusable
        el.setAttribute('tabindex', '-1');
        if (el instanceof HTMLButtonElement || 
            el instanceof HTMLInputElement || 
            el instanceof HTMLSelectElement || 
            el instanceof HTMLTextAreaElement || 
            el instanceof HTMLAnchorElement) {
          el.disabled = true;
        }
        
        // Check for autofocus
        observeAutofocus(el);
      });
      
      // Create a mutation observer to watch for changes
      const observer = new MutationObserver(mutations => {
        mutations.forEach(mutation => {
          if (mutation.type === 'childList') {
            // New elements were added, make them inert too
            mutation.addedNodes.forEach(node => {
              if (node.nodeType === Node.ELEMENT_NODE) {
                recursivelyCheckForInert(node);
              }
            });
          }
        });
      });
      
      // Start observing the inert element
      observer.observe(inertRoot, { childList: true, subtree: true });
      
      // Store the observer on the element
      inertRoot._inertObserver = observer;
    } else {
      // If the element is not inert, restore original state
      if (element._inertObserver) {
        element._inertObserver.disconnect();
        delete element._inertObserver;
      }
      
      // Restore original state for all elements
      const elementsToRestore = Array.from(element.querySelectorAll('*'));
      elementsToRestore.forEach(el => {
        if (el.hasAttribute('data-original-tabindex')) {
          el.setAttribute('tabindex', el.getAttribute('data-original-tabindex'));
          el.removeAttribute('data-original-tabindex');
        } else if (el.hasAttribute('tabindex') && el.getAttribute('tabindex') === '-1') {
          el.removeAttribute('tabindex');
        }
        
        if (el.hasAttribute('data-original-disabled')) {
          el.removeAttribute('disabled');
          el.removeAttribute('data-original-disabled');
        } else if (el.disabled && 
                  (el instanceof HTMLButtonElement || 
                   el instanceof HTMLInputElement || 
                   el instanceof HTMLSelectElement || 
                   el instanceof HTMLTextAreaElement || 
                   el instanceof HTMLAnchorElement)) {
          el.disabled = false;
        }
      });
    }
  }

  // Create a mutation observer to watch for inert attribute changes
  const observer = new MutationObserver(mutations => {
    mutations.forEach(mutation => {
      if (mutation.type === 'attributes' && mutation.attributeName === 'inert') {
        recursivelyCheckForInert(mutation.target);
      }
    });
  });

  // Start observing the document
  observer.observe(document.body, { 
    attributes: true, 
    attributeFilter: ['inert'],
    subtree: true
  });

  // Initial check for any elements that already have inert
  document.querySelectorAll('[inert]').forEach(el => {
    recursivelyCheckForInert(el);
  });
}
