<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\RecordAttachment;

class CleanupOrphanedAttachments extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'attachments:cleanup {--dry-run : Show what would be deleted without actually deleting}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Clean up orphaned attachment records (database records without actual files)';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $isDryRun = $this->option('dry-run');

        $this->info('Checking for orphaned attachment records...');

        $attachments = RecordAttachment::all();
        $orphanedCount = 0;
        $orphanedAttachments = [];

        foreach ($attachments as $attachment) {
            $filePath = storage_path('app/public/uploads/' . $attachment->file_path);

            if (!file_exists($filePath)) {
                $orphanedCount++;
                $orphanedAttachments[] = $attachment;

                $this->warn("Missing file: {$attachment->file_name} (ID: {$attachment->id}, Record: {$attachment->record_id})");
            }
        }

        if ($orphanedCount === 0) {
            $this->info('No orphaned attachment records found.');
            return;
        }

        $this->info("Found {$orphanedCount} orphaned attachment records.");

        if ($isDryRun) {
            $this->info('Dry run mode - no records will be deleted.');
            $this->info('Run without --dry-run to actually delete these records.');
            return;
        }

        if ($this->confirm("Do you want to delete these {$orphanedCount} orphaned attachment records?")) {
            foreach ($orphanedAttachments as $attachment) {
                $attachment->delete();
                $this->line("Deleted: {$attachment->file_name} (ID: {$attachment->id})");
            }

            $this->info("Successfully deleted {$orphanedCount} orphaned attachment records.");
        } else {
            $this->info('Operation cancelled.');
        }
    }
}
