<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\VesselRecord;
use App\Models\Vessel;
use App\Models\Setting;

class DashboardController extends Controller
{
    public function index(Request $request)
    {
        $searchTerm = $request->get('search', '');
        
        // Get warning and critical days from settings
        $warningDays = (int) Setting::getValue('warning_days', 8);
        $criticalDays = (int) Setting::getValue('critical_days', 5);
        
        // Build base query with search filter
        $baseQuery = function() use ($searchTerm) {
            $query = VesselRecord::with(['vessel', 'status', 'attachments']);

            // Add search filter if provided
            if (!empty($searchTerm)) {
                $query->where(function($q) use ($searchTerm) {
                    $q->where('invoice_no', 'like', "%{$searchTerm}%")
                      ->orWhere('port', 'like', "%{$searchTerm}%")
                      ->orWhere('description', 'like', "%{$searchTerm}%")
                      ->orWhere('company', 'like', "%{$searchTerm}%")
                      ->orWhereHas('vessel', function($subQuery) use ($searchTerm) {
                          $subQuery->where('name', 'like', "%{$searchTerm}%");
                      })
                      ->orWhereHas('status', function($subQuery) use ($searchTerm) {
                          $subQuery->where('name', 'like', "%{$searchTerm}%");
                      });
                });
            }

            return $query;
        };

        // Get ALL unpaid records (exclude paid status records)
        $records = $baseQuery()
            ->where(function($q) {
                $q->whereHas('status', function($subQ) {
                    $subQ->where('is_paid_status', false);
                })->orWhereNull('status_id');
            })
            ->get()
            ->sortBy(function($record) {
                return $record->due_date ? $record->due_date : '9999-12-31'; // Put records without due date at end
            });

        // Add due date status to each record
        $records = $records->map(function($record) use ($warningDays, $criticalDays) {
            if ($record->due_date) {
                $dueDate = \Carbon\Carbon::parse($record->due_date);
                $today = \Carbon\Carbon::today();
                $diffInDays = $today->diffInDays($dueDate, false);

                if ($diffInDays < 0) {
                    $record->due_date_status = 'overdue';
                } elseif ($diffInDays <= $criticalDays) {
                    $record->due_date_status = 'critical';
                } elseif ($diffInDays <= $warningDays) {
                    $record->due_date_status = 'warning';
                } else {
                    $record->due_date_status = 'normal';
                }
            } else {
                $record->due_date_status = 'normal';
            }
            return $record;
        });
        
        // Get all vessels for sidebar
        $vessels = Vessel::withCount('records')->orderBy('name')->get();

        // Get summary statistics
        $totalVessels = Vessel::count();
        $totalRecords = VesselRecord::count();
        $totalAmount = VesselRecord::sum('amount') ?? 0;

        // Get counts by status (from all unpaid records, not just the recent ones shown)
        $allUnpaidRecords = VesselRecord::with(['status'])
            ->where(function($q) {
                $q->whereHas('status', function($subQ) {
                    $subQ->where('is_paid_status', false);
                })->orWhereNull('status_id');
            })
            ->whereNotNull('due_date')
            ->get();

        // Add due date status to unpaid records for counting
        $allUnpaidRecords = $allUnpaidRecords->map(function($record) use ($warningDays, $criticalDays) {
            if ($record->due_date) {
                $dueDate = \Carbon\Carbon::parse($record->due_date);
                $today = \Carbon\Carbon::today();
                $diffInDays = $today->diffInDays($dueDate, false);

                if ($diffInDays < 0) {
                    $record->due_date_status = 'overdue';
                } elseif ($diffInDays <= $criticalDays) {
                    $record->due_date_status = 'critical';
                } elseif ($diffInDays <= $warningDays) {
                    $record->due_date_status = 'warning';
                } else {
                    $record->due_date_status = 'normal';
                }
            } else {
                $record->due_date_status = 'normal';
            }
            return $record;
        });

        $overdueCount = $allUnpaidRecords->where('due_date_status', 'overdue')->count();
        $criticalCount = $allUnpaidRecords->where('due_date_status', 'critical')->count();
        $warningCount = $allUnpaidRecords->where('due_date_status', 'warning')->count();

        return view('dashboard', compact(
            'records',
            'vessels',
            'searchTerm',
            'warningDays',
            'criticalDays',
            'totalVessels',
            'totalRecords',
            'totalAmount',
            'overdueCount',
            'criticalCount',
            'warningCount'
        ));
    }
}
