<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Vessel;
use App\Models\VesselRecord;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use PhpOffice\PhpSpreadsheet\Style\Font;
use Symfony\Component\HttpFoundation\StreamedResponse;

class ExportController extends Controller
{
    public function exportAll()
    {
        try {
            // Create new Spreadsheet object
            $spreadsheet = new Spreadsheet();
            // Remove default sheet
            $spreadsheet->removeSheetByIndex(0);

            // Fetch all vessels with their records
            $vessels = Vessel::with(['records.status'])->orderBy('name')->get();

            $sheetIndex = 0;
            foreach ($vessels as $vessel) {
                // Create a new worksheet for each vessel
                $worksheet = $spreadsheet->createSheet($sheetIndex);
                $worksheet->setTitle($this->sanitizeSheetName($vessel->name));

                // Set headers
                $headers = ['Date', 'Invoice No', 'Port', 'Description', 'Amount', 'Due Date', 'Company', 'Status'];
                $worksheet->fromArray($headers, null, 'A1');

                // Style headers
                $headerRange = 'A1:H1';
                $worksheet->getStyle($headerRange)->getFont()->setBold(true);
                $worksheet->getStyle($headerRange)->getFill()
                    ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
                    ->getStartColor()->setRGB('E0E0E0');

                // Add vessel records data
                $row = 2;
                foreach ($vessel->records as $record) {
                    $worksheet->fromArray([
                        $record->date,
                        $record->invoice_no,
                        $record->port,
                        $record->description,
                        $record->amount,
                        $record->due_date,
                        $record->company,
                        $record->status ? $record->status->name : ''
                    ], null, "A$row");

                    // Format amount column as currency
                    if ($record->amount) {
                        $worksheet->getStyle("E$row")->getNumberFormat()
                            ->setFormatCode('$#,##0.00');
                    }

                    $row++;
                }

                // Auto-size columns
                foreach (range('A', 'H') as $col) {
                    $worksheet->getColumnDimension($col)->setAutoSize(true);
                }

                $sheetIndex++;
            }

            // If no vessels found, create a single sheet with message
            if ($vessels->isEmpty()) {
                $worksheet = $spreadsheet->createSheet(0);
                $worksheet->setTitle('No Data');
                $worksheet->setCellValue('A1', 'No vessels found');
            }

            // Generate filename with current date
            $filename = 'bunker_export_' . date('Y-m-d_H-i-s') . '.xlsx';

            // Create a streamed response
            return new StreamedResponse(function() use ($spreadsheet) {
                $writer = new Xlsx($spreadsheet);
                $writer->save('php://output');
            }, 200, [
                'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                'Content-Disposition' => 'attachment; filename="' . $filename . '"',
                'Cache-Control' => 'max-age=0',
            ]);

        } catch (\Exception $e) {
            return back()->with('error', 'Export failed: ' . $e->getMessage());
        }
    }

    public function exportReport(Request $request)
    {
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'vessel_id' => 'nullable|exists:vessels,id',
            'format' => 'required|in:excel,pdf'
        ]);

        $query = VesselRecord::with(['vessel', 'status'])
            ->whereBetween('date', [$request->start_date, $request->end_date]);

        if ($request->vessel_id) {
            $query->where('vessel_id', $request->vessel_id);
        }

        $records = $query->orderBy('date')->get();

        if ($request->format === 'excel') {
            return $this->exportReportExcel($records, $request->start_date, $request->end_date);
        } else {
            return $this->exportReportPdf($records, $request->start_date, $request->end_date);
        }
    }

    private function exportReportExcel($records, $startDate, $endDate)
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Bunker Report');

        // Set headers
        $headers = ['Vessel', 'Date', 'Invoice No', 'Port', 'Description', 'Amount', 'Due Date', 'Company', 'Status'];
        $sheet->fromArray($headers, null, 'A1');

        // Style headers
        $headerRange = 'A1:I1';
        $sheet->getStyle($headerRange)->getFont()->setBold(true);
        $sheet->getStyle($headerRange)->getFill()
            ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
            ->getStartColor()->setRGB('E0E0E0');

        // Add data
        $row = 2;
        $totalAmount = 0;
        foreach ($records as $record) {
            $sheet->fromArray([
                $record->vessel->name,
                $record->date,
                $record->invoice_no,
                $record->port,
                $record->description,
                $record->amount,
                $record->due_date,
                $record->company,
                $record->status ? $record->status->name : ''
            ], null, "A$row");

            // Format amount column as currency
            if ($record->amount) {
                $sheet->getStyle("F$row")->getNumberFormat()
                    ->setFormatCode('$#,##0.00');
                $totalAmount += $record->amount;
            }

            $row++;
        }

        // Add total row
        $sheet->setCellValue("E$row", 'Total:');
        $sheet->setCellValue("F$row", $totalAmount);
        $sheet->getStyle("E$row:F$row")->getFont()->setBold(true);
        $sheet->getStyle("F$row")->getNumberFormat()
            ->setFormatCode('$#,##0.00');

        // Auto-size columns
        foreach (range('A', 'I') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }

        $filename = 'bunker_report_' . $startDate . '_to_' . $endDate . '.xlsx';

        return new StreamedResponse(function() use ($spreadsheet) {
            $writer = new Xlsx($spreadsheet);
            $writer->save('php://output');
        }, 200, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
            'Cache-Control' => 'max-age=0',
        ]);
    }

    private function exportReportPdf($records, $startDate, $endDate)
    {
        // For now, return Excel format as PDF export would require additional setup
        return $this->exportReportExcel($records, $startDate, $endDate);
    }

    private function sanitizeSheetName($name)
    {
        // Remove invalid characters for Excel sheet names
        $name = preg_replace('/[\\\\\/\?\*\[\]:]/', '', $name);
        // Limit to 31 characters (Excel limit)
        return substr($name, 0, 31);
    }
}
