<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\StatusOption;
use App\Models\Setting;

class SettingController extends Controller
{
    public function index()
    {
        $statuses = StatusOption::orderBy('name')->get();
        $settings = Setting::pluck('setting_value', 'setting_name')->toArray();
        return view('settings.index', compact('statuses', 'settings'));
    }

    public function update(Request $request)
    {
        $request->validate([
            'warning_days' => 'required|integer|min:1',
            'critical_days' => 'required|integer|min:1',
        ]);

        // Update or create settings
        Setting::updateOrCreate(
            ['setting_name' => 'warning_days'],
            ['setting_value' => $request->warning_days]
        );

        Setting::updateOrCreate(
            ['setting_name' => 'critical_days'],
            ['setting_value' => $request->critical_days]
        );

        return redirect()->route('settings.index')->with('success', 'Settings updated successfully!');
    }

    public function storeStatus(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:status_options,name',
        ]);

        $status = StatusOption::create([
            'name' => $request->name,
            'is_paid_status' => false,
        ]);

        return response()->json([
            'success' => true,
            'status' => $status,
            'message' => 'Status created successfully!'
        ]);
    }

    public function updateStatus(Request $request, StatusOption $status)
    {
        $data = [];

        if ($request->has('name')) {
            $request->validate([
                'name' => 'required|string|max:255|unique:status_options,name,' . $status->id,
            ]);
            $data['name'] = $request->name;
        }

        if ($request->has('is_paid_status')) {
            $data['is_paid_status'] = $request->boolean('is_paid_status');
        }

        $status->update($data);

        return response()->json([
            'success' => true,
            'status' => $status,
            'message' => 'Status updated successfully!'
        ]);
    }

    public function destroyStatus(StatusOption $status)
    {
        // Check if status is being used by any records
        if ($status->records()->count() > 0) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete status that is being used by records.'
            ], 400);
        }

        $status->delete();

        return response()->json([
            'success' => true,
            'message' => 'Status deleted successfully!'
        ]);
    }
}
