<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\VesselRecord;
use App\Models\Vessel;
use App\Models\StatusOption;
use App\Models\RecordAttachment;

class VesselRecordController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $records = VesselRecord::with(['vessel', 'status', 'attachments'])
                              ->orderBy('due_date')
                              ->paginate(15);
        return view('vessel-records.index', compact('records'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $vessels = Vessel::orderBy('name')->get();
        $statuses = StatusOption::orderBy('name')->get();
        $selectedVesselId = $request->get('vessel_id');
        return view('vessel-records.create', compact('vessels', 'statuses', 'selectedVesselId'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'vessel_id' => 'required|exists:vessels,id',
            'invoice_no' => 'required|string|max:50',
            'port' => 'nullable|string|max:100',
            'description' => 'nullable|string',
            'amount' => 'nullable|numeric|min:0',
            'due_date' => 'nullable|string',
            'company' => 'nullable|string|max:100',
            'status_id' => 'nullable|exists:status_options,id',
            'attachments.*' => 'nullable|file|max:10240|mimes:pdf,doc,docx,xls,xlsx,jpg,jpeg,png,txt',
        ]);

        $data = $request->all();
        $data['created_by'] = auth()->id();

        // Convert DD/MM/YYYY format to YYYY-MM-DD for database storage
        if (!empty($data['date'])) {
            $data['date'] = $this->convertDateFormat($data['date']);
        }
        if (!empty($data['due_date'])) {
            $data['due_date'] = $this->convertDateFormat($data['due_date']);
        }

        $record = VesselRecord::create($data);

        // Handle file uploads
        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $fileName = $file->getClientOriginalName();
                $filePath = time() . '_' . $fileName;

                // Store file in storage/app/public/uploads (consistent with AttachmentController)
                $file->storeAs('uploads', $filePath, 'public');

                // Create attachment record
                RecordAttachment::create([
                    'record_id' => $record->id,
                    'file_name' => $fileName,
                    'file_path' => $filePath,
                    'uploaded_at' => now(),
                ]);
            }
        }

        return redirect()->route('vessel-records.index')->with('success', 'Record created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(VesselRecord $vesselRecord)
    {
        $vesselRecord->load(['vessel', 'status', 'attachments']);
        return view('vessel-records.show', compact('vesselRecord'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(VesselRecord $vesselRecord)
    {
        $vessels = Vessel::orderBy('name')->get();
        $statuses = StatusOption::orderBy('name')->get();
        return view('vessel-records.edit', compact('vesselRecord', 'vessels', 'statuses'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, VesselRecord $vesselRecord)
    {
        $request->validate([
            'vessel_id' => 'required|exists:vessels,id',
            'invoice_no' => 'required|string|max:50',
            'port' => 'nullable|string|max:100',
            'description' => 'nullable|string',
            'amount' => 'nullable|numeric|min:0',
            'due_date' => 'nullable|string',
            'company' => 'nullable|string|max:100',
            'status_id' => 'nullable|exists:status_options,id',
            'attachments.*' => 'nullable|file|max:10240', // 10MB max per file
        ]);

        $data = $request->all();

        // Convert DD/MM/YYYY format to YYYY-MM-DD for database storage
        if (!empty($data['date'])) {
            $data['date'] = $this->convertDateFormat($data['date']);
        }
        if (!empty($data['due_date'])) {
            $data['due_date'] = $this->convertDateFormat($data['due_date']);
        }

        $vesselRecord->update($data);

        // Handle new file uploads
        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $originalName = $file->getClientOriginalName();
                $fileName = time() . '_' . $originalName;
                $filePath = $file->storeAs('uploads', $fileName, 'public');

                RecordAttachment::create([
                    'record_id' => $vesselRecord->id,
                    'file_name' => $originalName,
                    'file_path' => $fileName,
                    'uploaded_at' => now(),
                ]);
            }
        }

        return redirect()->route('vessel-records.show', $vesselRecord)->with('success', 'Record updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(VesselRecord $vesselRecord)
    {
        $vesselRecord->delete();
        return redirect()->route('vessel-records.index')->with('success', 'Record deleted successfully.');
    }

    /**
     * Show paid records
     */
    public function paidRecords(Request $request)
    {
        $searchTerm = $request->get('search', '');

        $query = VesselRecord::with(['vessel', 'status', 'attachments'])
            ->whereHas('status', function($q) {
                $q->where('is_paid_status', true);
            });

        // Handle search
        if ($searchTerm) {
            $query->where(function($q) use ($searchTerm) {
                $q->where('invoice_no', 'like', "%{$searchTerm}%")
                  ->orWhere('port', 'like', "%{$searchTerm}%")
                  ->orWhere('company', 'like', "%{$searchTerm}%")
                  ->orWhere('description', 'like', "%{$searchTerm}%")
                  ->orWhereHas('vessel', function($subQ) use ($searchTerm) {
                      $subQ->where('name', 'like', "%{$searchTerm}%");
                  });
            });
        }

        $records = $query->orderBy('created_at', 'desc')->paginate(20);

        return view('paid-records.index', compact('records', 'searchTerm'));
    }

    /**
     * Convert DD/MM/YYYY format to YYYY-MM-DD format for database storage
     */
    private function convertDateFormat($dateString)
    {
        if (empty($dateString)) {
            return null;
        }

        // Check if it's already in YYYY-MM-DD format
        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $dateString)) {
            return $dateString;
        }

        // Convert DD/MM/YYYY to YYYY-MM-DD
        if (preg_match('/^(\d{1,2})\/(\d{1,2})\/(\d{4})$/', $dateString, $matches)) {
            $day = str_pad($matches[1], 2, '0', STR_PAD_LEFT);
            $month = str_pad($matches[2], 2, '0', STR_PAD_LEFT);
            $year = $matches[3];
            return "$year-$month-$day";
        }

        // If format doesn't match, try to parse with Carbon
        try {
            return \Carbon\Carbon::createFromFormat('d/m/Y', $dateString)->format('Y-m-d');
        } catch (\Exception $e) {
            return null;
        }
    }
}
