<?php

// This script checks the database for vessels with 0 certificates

// Load the Laravel environment
require __DIR__.'/vendor/autoload.php';
$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

use App\Models\Vessel;
use App\Models\VesselCertificate;
use Illuminate\Support\Facades\DB;

echo "Checking vessels for certificates...\n";

$vessels = Vessel::all();
echo "Found " . $vessels->count() . " vessels\n";

$vesselsWithZeroCertificates = 0;

foreach ($vessels as $vessel) {
    $certificateCount = VesselCertificate::where('vessel_id', $vessel->id)->count();
    
    echo "Vessel: " . $vessel->name . " (ID: " . $vessel->id . ") - Certificates: " . $certificateCount . "\n";
    
    if ($certificateCount === 0) {
        $vesselsWithZeroCertificates++;
    }
}

echo "\nVessels with 0 certificates: " . $vesselsWithZeroCertificates . "\n";

// Check if there are any certificates in the database
$totalCertificates = VesselCertificate::count();
echo "Total certificates in database: " . $totalCertificates . "\n";

// Check if there are any certificates in the SQL file
$sqlFilePath = __DIR__ . '/aikhrrte_certificates.sql';
if (file_exists($sqlFilePath)) {
    $sqlContent = file_get_contents($sqlFilePath);
    
    // Extract vessel_certificates data
    preg_match('/INSERT INTO `vessel_certificates` \(`id`, `vessel_id`, `certificate_id`, `id_no`, `vessel_department`, `issued_date`, `expiry_date`, `attachment_path`, `created_at`, `updated_at`\) VALUES(.*?);/s', $sqlContent, $matches);
    
    if (!empty($matches)) {
        $valuesString = $matches[1];
        
        // Parse values
        preg_match_all('/\((.*?)\)(?:,|$)/s', $valuesString, $valueMatches);
        $values = $valueMatches[1];
        
        echo "Certificates in SQL file: " . count($values) . "\n";
        
        // Count certificates per vessel in SQL file
        $sqlCertificatesPerVessel = [];
        
        foreach ($values as $value) {
            $fields = str_getcsv($value, ',', "'");
            
            // Clean up fields
            foreach ($fields as &$field) {
                $field = trim($field);
                if ($field === 'NULL' || $field === 'null') {
                    $field = null;
                }
            }
            
            // Extract vessel ID
            $vesselId = (int)$fields[1];
            
            if (!isset($sqlCertificatesPerVessel[$vesselId])) {
                $sqlCertificatesPerVessel[$vesselId] = 0;
            }
            
            $sqlCertificatesPerVessel[$vesselId]++;
        }
        
        echo "\nCertificates per vessel in SQL file:\n";
        foreach ($sqlCertificatesPerVessel as $vesselId => $count) {
            $vessel = Vessel::find($vesselId);
            $vesselName = $vessel ? $vessel->name : "Unknown";
            echo "Vessel ID: " . $vesselId . " (" . $vesselName . ") - Certificates: " . $count . "\n";
        }
    } else {
        echo "No vessel_certificates data found in SQL file\n";
    }
} else {
    echo "SQL file not found at: " . $sqlFilePath . "\n";
}

// Check for any database issues
echo "\nChecking for database issues...\n";

// Check for orphaned certificates
$orphanedCertificates = VesselCertificate::whereNotIn('vessel_id', Vessel::pluck('id'))->count();
echo "Orphaned certificates (no vessel): " . $orphanedCertificates . "\n";

// Check for certificates with invalid certificate_id
$invalidCertificates = VesselCertificate::whereNotIn('certificate_id', DB::table('certificates')->pluck('id'))->count();
echo "Invalid certificates (no certificate type): " . $invalidCertificates . "\n";

echo "\nDone!\n";
