<?php
// This script imports certificates from the all_certificates.txt file

// Load the Laravel environment
require __DIR__.'/vendor/autoload.php';
$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

use App\Models\Vessel;
use App\Models\Certificate;
use App\Models\VesselCertificate;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

// Function to parse SQL file and extract vessel certificates
function extractVesselCertificatesFromSql($sqlFilePath) {
    if (!file_exists($sqlFilePath)) {
        echo "SQL file not found at: $sqlFilePath\n";
        return [];
    }

    $sqlContent = file_get_contents($sqlFilePath);

    // Extract all the certificate data directly from the file
    // Format: (id, vessel_id, certificate_id, 'id_no', 'vessel_department', 'issued_date', 'expiry_date', 'attachment_path', 'created_at', 'updated_at')
    preg_match_all('/\((\d+),\s*(\d+),\s*(\d+),\s*(.*?),\s*(.*?),\s*(.*?),\s*(.*?),\s*(.*?),\s*(.*?),\s*(.*?)\)(?:,|$)/s', $sqlContent, $matches, PREG_SET_ORDER);

    $certificates = [];

    foreach ($matches as $match) {
        // Skip if not enough fields
        if (count($match) < 11) {
            echo "Skipping row with insufficient fields: " . count($match) . "\n";
            continue;
        }

        $id = (int)$match[1];
        $vesselId = (int)$match[2];
        $certificateId = (int)$match[3];

        // Clean up id_no
        $idNo = trim($match[4], "'");
        if ($idNo === 'NULL' || $idNo === 'null') $idNo = null;

        // Clean up vessel_department
        $vesselDepartment = trim($match[5], "'");
        if ($vesselDepartment === 'NULL' || $vesselDepartment === 'null') $vesselDepartment = null;

        // Clean up issued_date
        $issuedDate = trim($match[6], "'");
        if ($issuedDate === 'NULL' || $issuedDate === 'null' || $issuedDate === '0000-00-00') $issuedDate = null;

        // Clean up expiry_date
        $expiryDate = trim($match[7], "'");
        if ($expiryDate === 'NULL' || $expiryDate === 'null' || $expiryDate === '0000-00-00') $expiryDate = null;

        // Clean up attachment_path
        $attachmentPath = trim($match[8], "'");
        if ($attachmentPath === 'NULL' || $attachmentPath === 'null') {
            $attachmentPath = null;
        } else {
            // Clean up attachment path
            $attachmentPath = str_replace('\\', '', $attachmentPath);

            // Check if it's a JSON array
            if (substr($attachmentPath, 0, 1) === '[' && substr($attachmentPath, -1) === ']') {
                // It's already a JSON string, keep it as is
            } else {
                // Convert to JSON array format
                $attachmentPath = json_encode([$attachmentPath]);
            }
        }

        // Clean up created_at and updated_at
        $createdAt = trim($match[9], "'");
        $updatedAt = trim($match[10], "'");

        $certificates[] = [
            'id' => $id,
            'vessel_id' => $vesselId,
            'certificate_id' => $certificateId,
            'id_no' => $idNo,
            'vessel_department' => $vesselDepartment,
            'issued_date' => $issuedDate,
            'expiry_date' => $expiryDate,
            'attachment_path' => $attachmentPath,
            'created_at' => $createdAt,
            'updated_at' => $updatedAt,
        ];
    }

    return $certificates;
}

// Main execution
echo "Starting certificate import from all_certificates.txt...\n";

// Get all vessels
$vessels = Vessel::all();
echo "Found " . $vessels->count() . " vessels\n";

// Get all certificates
$certificates = Certificate::all();
echo "Found " . $certificates->count() . " certificate types\n";

// Extract certificates from SQL file
$sqlFilePath = __DIR__ . '/all_certificates.txt';
$sqlCertificates = extractVesselCertificatesFromSql($sqlFilePath);
echo "Extracted " . count($sqlCertificates) . " certificates from SQL file\n";

// Group certificates by vessel
$certificatesByVessel = [];
foreach ($sqlCertificates as $cert) {
    $vesselId = $cert['vessel_id'];
    if (!isset($certificatesByVessel[$vesselId])) {
        $certificatesByVessel[$vesselId] = [];
    }
    $certificatesByVessel[$vesselId][] = $cert;
}

// Begin transaction
DB::beginTransaction();

try {
    $importCount = 0;
    $updateCount = 0;
    $skipCount = 0;

    // First, make sure all vessels have all certificate types
    foreach ($vessels as $vessel) {
        $vesselId = $vessel->id;

        // Create empty certificates for all certificate types that don't exist yet
        foreach ($certificates as $certificate) {
            $exists = VesselCertificate::where('vessel_id', $vesselId)
                ->where('certificate_id', $certificate->id)
                ->exists();

            if (!$exists) {
                VesselCertificate::create([
                    'vessel_id' => $vesselId,
                    'certificate_id' => $certificate->id,
                ]);

                $importCount++;
                echo "  Created empty certificate for vessel {$vessel->name}, certificate {$certificate->name}\n";
            }
        }
    }

    // Now import the certificate data from the SQL file
    foreach ($sqlCertificates as $cert) {
        $vesselId = $cert['vessel_id'];
        $certificateId = $cert['certificate_id'];

        // Find the existing certificate
        $existingCert = VesselCertificate::where('vessel_id', $vesselId)
            ->where('certificate_id', $certificateId)
            ->first();

        if ($existingCert) {
            // Update the existing certificate with data from SQL
            $existingCert->id_no = $cert['id_no'];
            $existingCert->vessel_department = $cert['vessel_department'];

            // Validate dates before setting them
            if ($cert['issued_date'] && \DateTime::createFromFormat('Y-m-d', $cert['issued_date'])) {
                $existingCert->issued_date = $cert['issued_date'];
            }

            if ($cert['expiry_date'] && \DateTime::createFromFormat('Y-m-d', $cert['expiry_date'])) {
                $existingCert->expiry_date = $cert['expiry_date'];
            }

            $existingCert->attachment_path = $cert['attachment_path'];
            $existingCert->save();

            $updateCount++;
            echo "  Updated certificate for vessel ID {$vesselId}, certificate ID {$certificateId}\n";
        } else {
            $skipCount++;
            echo "  Certificate not found for vessel ID {$vesselId}, certificate ID {$certificateId}\n";
        }
    }

    // Commit transaction
    DB::commit();

    echo "Import completed successfully!\n";
    echo "Created: {$importCount}, Updated: {$updateCount}, Skipped: {$skipCount}\n";

} catch (\Exception $e) {
    // Rollback transaction on error
    DB::rollBack();

    echo "Error during import: " . $e->getMessage() . "\n";
    echo $e->getTraceAsString() . "\n";
}
