@extends('layouts.app')

@section('content')
<div class="container-fluid">
    <div class="row mb-3">
        <div class="col-md-12">
            <h2>Certificate Management</h2>
            <p class="text-muted">Manage certificates, add new certificates to all vessels, and reorder certificates.</p>
        </div>
    </div>

    @if(session('success'))
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        {{ session('success') }}
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
    @endif

    @if(session('error'))
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        {{ session('error') }}
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
    @endif

    <div class="row">
        <div class="col-md-12 mb-4">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">Create New Certificate</h5>
                </div>
                <div class="card-body">
                    <form action="{{ route('settings.create-certificate') }}" method="POST" class="row g-3">
                        @csrf
                        <div class="col-md-6">
                            <label for="name" class="form-label">Certificate Name</label>
                            <input type="text" class="form-control @error('name') is-invalid @enderror" id="name" name="name" required>
                            @error('name')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                        <div class="col-md-6">
                            <label for="category" class="form-label">Category</label>
                            <input type="text" class="form-control @error('category') is-invalid @enderror" id="category" name="category" required list="categories">
                            <datalist id="categories">
                                @foreach($categories as $category)
                                    <option value="{{ $category }}">
                                @endforeach
                            </datalist>
                            @error('category')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                        <div class="col-md-12">
                            <small class="text-muted">New certificates are automatically added to all vessels.</small>
                        </div>
                        <div class="col-md-12">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-plus-circle"></i> Create Certificate
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <div class="col-md-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">Certificate Reordering</h5>
                    <div>
                        <button id="saveOrderBtn" class="btn btn-success btn-sm" style="display: none;">
                            <i class="fas fa-save"></i> Save Order
                        </button>
                    </div>
                </div>
                <div class="card-body">
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle"></i> Drag and drop certificates to reorder them. Changes will be applied to all vessels.
                    </div>

                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead>
                                <tr>
                                    <th width="5%">#</th>
                                    <th width="25%">Category</th>
                                    <th width="40%">Certificate Name</th>
                                    <th width="10%">Display Order</th>
                                    <th width="20%">Actions</th>
                                </tr>
                            </thead>
                            <tbody id="certificatesTableBody">
                                @foreach($certificates->groupBy('category') as $category => $categoryCertificates)
                                    @foreach($categoryCertificates->sortBy('display_order') as $certificate)
                                        <tr class="certificate-row" data-id="{{ $certificate->id }}" data-order="{{ $certificate->display_order }}">
                                            <td>{{ $loop->parent->index + 1 }}.{{ $loop->index + 1 }}</td>
                                            <td>{{ $category }}</td>
                                            <td>{{ $certificate->name }}</td>
                                            <td>
                                                <span class="order-display">{{ $certificate->display_order }}</span>
                                                <input type="hidden" class="order-input" value="{{ $certificate->display_order }}">
                                            </td>
                                            <td>
                                                <div class="btn-group" role="group">
                                                    <button type="button" class="btn btn-sm btn-warning edit-certificate-btn"
                                                            data-id="{{ $certificate->id }}"
                                                            data-name="{{ $certificate->name }}"
                                                            data-category="{{ $certificate->category }}">
                                                        <i class="fas fa-edit"></i> Edit
                                                    </button>
                                                    <button type="button" class="btn btn-sm btn-danger delete-certificate-btn"
                                                            data-id="{{ $certificate->id }}"
                                                            data-name="{{ $certificate->name }}">
                                                        <i class="fas fa-trash"></i> Delete
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    @endforeach
                                @endforeach
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Edit Certificate Modal -->
<div class="modal fade" id="editCertificateModal" tabindex="-1" aria-labelledby="editCertificateModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="editCertificateModalLabel">Edit Certificate</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form id="editCertificateForm" method="POST">
                @csrf
                @method('PUT')
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="edit_name" class="form-label">Certificate Name</label>
                        <input type="text" class="form-control" id="edit_name" name="name" required>
                    </div>
                    <div class="mb-3">
                        <label for="edit_category" class="form-label">Category</label>
                        <input type="text" class="form-control" id="edit_category" name="category" required list="edit_categories">
                        <datalist id="edit_categories">
                            @foreach($categories as $category)
                                <option value="{{ $category }}">
                            @endforeach
                        </datalist>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Save Changes</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Delete Certificate Confirmation Modal -->
<div class="modal fade" id="deleteCertificateModal" tabindex="-1" aria-labelledby="deleteCertificateModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="deleteCertificateModalLabel">Delete Certificate</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div id="deleteCertificateAlerts"></div>
                <p>Are you sure you want to delete the certificate: <strong id="delete_certificate_name"></strong>?</p>
                <p class="text-danger">This will remove the certificate from all vessels and cannot be undone.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" id="confirmDeleteCertificateBtn" class="btn btn-danger">Delete Certificate</button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script src="https://cdn.jsdelivr.net/npm/sortablejs@1.14.0/Sortable.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Initialize Sortable for the certificates table
    const certificatesTableBody = document.getElementById('certificatesTableBody');
    const saveOrderBtn = document.getElementById('saveOrderBtn');
    const editCertificateModal = new bootstrap.Modal(document.getElementById('editCertificateModal'));
    const deleteCertificateModal = new bootstrap.Modal(document.getElementById('deleteCertificateModal'));

    let sortable = new Sortable(certificatesTableBody, {
        animation: 150,
        handle: '.certificate-row',
        ghostClass: 'bg-light',
        onEnd: function() {
            // Update order numbers
            updateOrderNumbers();
            // Show save button
            saveOrderBtn.style.display = 'inline-block';
        }
    });

    // Function to update order numbers
    function updateOrderNumbers() {
        const rows = certificatesTableBody.querySelectorAll('.certificate-row');
        rows.forEach((row, index) => {
            const orderDisplay = row.querySelector('.order-display');
            const orderInput = row.querySelector('.order-input');

            orderDisplay.textContent = index + 1;
            orderInput.value = index + 1;
        });
    }

    // Save order button click handler
    saveOrderBtn.addEventListener('click', function() {
        const rows = certificatesTableBody.querySelectorAll('.certificate-row');
        const certificates = [];

        rows.forEach(row => {
            certificates.push({
                id: row.dataset.id,
                display_order: row.querySelector('.order-input').value
            });
        });

        // Send AJAX request to save order
        fetch('{{ route("settings.reorder-certificates") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            },
            body: JSON.stringify({ certificates: certificates })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Show success message
                alert('Certificate order saved successfully!');
                // Hide save button
                saveOrderBtn.style.display = 'none';
            } else {
                alert('Error: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('An error occurred while saving the certificate order.');
        });
    });

    // Edit Certificate Button Click Handler
    const editButtons = document.querySelectorAll('.edit-certificate-btn');
    editButtons.forEach(button => {
        button.addEventListener('click', function() {
            const certificateId = this.getAttribute('data-id');
            const certificateName = this.getAttribute('data-name');
            const certificateCategory = this.getAttribute('data-category');

            // Set form action
            document.getElementById('editCertificateForm').action = `/settings/certificates/${certificateId}`;

            // Set form values
            document.getElementById('edit_name').value = certificateName;
            document.getElementById('edit_category').value = certificateCategory;

            // Show modal
            editCertificateModal.show();
        });
    });

    // Delete Certificate Button Click Handler
    const deleteButtons = document.querySelectorAll('.delete-certificate-btn');
    deleteButtons.forEach(button => {
        button.addEventListener('click', function() {
            const certificateId = this.getAttribute('data-id');
            const certificateName = this.getAttribute('data-name');

            // Store certificate ID in a data attribute on the confirm button
            document.getElementById('confirmDeleteCertificateBtn').setAttribute('data-certificate-id', certificateId);

            // Set certificate name in confirmation message
            document.getElementById('delete_certificate_name').textContent = certificateName;

            // Clear any previous alerts
            document.getElementById('deleteCertificateAlerts').innerHTML = '';

            // Show modal
            deleteCertificateModal.show();
        });
    });

    // Confirm Delete Certificate Button Click Handler
    document.getElementById('confirmDeleteCertificateBtn').addEventListener('click', function() {
        const certificateId = this.getAttribute('data-certificate-id');
        const alertsContainer = document.getElementById('deleteCertificateAlerts');

        // Show loading state
        this.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Deleting...';
        this.disabled = true;

        // Send AJAX request to delete certificate
        fetch(`/settings/certificates/${certificateId}`, {
            method: 'DELETE',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.json();
        })
        .then(data => {
            if (data.success) {
                // Show success message
                alertsContainer.innerHTML = `
                    <div class="alert alert-success">
                        ${data.message || 'Certificate deleted successfully!'}
                    </div>
                `;

                // Remove the certificate row from the table
                const certificateRow = document.querySelector(`.certificate-row[data-id="${certificateId}"]`);
                if (certificateRow) {
                    certificateRow.remove();
                }

                // After 2 seconds, close the modal and reload the page
                setTimeout(() => {
                    deleteCertificateModal.hide();
                    window.location.reload(); // Reload to ensure everything is updated correctly
                }, 2000);
            } else {
                // Show error message
                alertsContainer.innerHTML = `
                    <div class="alert alert-danger">
                        ${data.message || 'An error occurred while deleting the certificate.'}
                    </div>
                `;

                // Reset button state
                this.innerHTML = 'Delete Certificate';
                this.disabled = false;
            }
        })
        .catch(error => {
            console.error('Error:', error);

            // Show error message
            alertsContainer.innerHTML = `
                <div class="alert alert-danger">
                    An error occurred while deleting the certificate. The certificate may be in use by vessels.
                </div>
            `;

            // Reset button state
            this.innerHTML = 'Delete Certificate';
            this.disabled = false;
        });
    });
});
</script>
@endsection
